<?php
// admin.php - Enhanced admin panel with improved security
session_start();

// Load config for better security
$CONFIG_FILE = __DIR__ . '/admin_config.json';
$config = json_decode(@file_get_contents($CONFIG_FILE) ?: '{}', true);
$ADMIN_PASSWORD = $config['admin_password'] ?? 'ReplaceThisWithAStrongPass';
$SESSION_TIMEOUT = $config['session_timeout'] ?? 3600; // 1 hour default

// Session timeout check
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $SESSION_TIMEOUT) {
    session_destroy();
    header("Location: " . $_SERVER['REQUEST_URI']);
    exit;
}
$_SESSION['last_activity'] = time();

// Enhanced login with rate limiting
$max_attempts = 5;
$lockout_time = 900; // 15 minutes
$attempts_file = __DIR__ . '/login_attempts.json';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['admin_pass'])) {
    $client_ip = $_SERVER['REMOTE_ADDR'];
    $attempts_data = json_decode(@file_get_contents($attempts_file) ?: '{}', true);
    
    // Check if IP is locked out
    if (isset($attempts_data[$client_ip])) {
        $attempts = $attempts_data[$client_ip];
        if ($attempts['count'] >= $max_attempts && (time() - $attempts['last_attempt']) < $lockout_time) {
            $remaining_time = $lockout_time - (time() - $attempts['last_attempt']);
            $login_error = "Too many failed attempts. Try again in " . ceil($remaining_time / 60) . " minutes.";
        }
    }
    
    if (empty($login_error)) {
        if (hash_equals($ADMIN_PASSWORD, $_POST['admin_pass'])) {
            $_SESSION['is_admin'] = true;
            $_SESSION['login_time'] = time();
            // Reset failed attempts on successful login
            unset($attempts_data[$client_ip]);
            file_put_contents($attempts_file, json_encode($attempts_data));
        } else {
            // Record failed attempt
            if (!isset($attempts_data[$client_ip])) {
                $attempts_data[$client_ip] = ['count' => 0, 'last_attempt' => 0];
            }
            $attempts_data[$client_ip]['count']++;
            $attempts_data[$client_ip]['last_attempt'] = time();
            file_put_contents($attempts_file, json_encode($attempts_data));
            
            $remaining_attempts = $max_attempts - $attempts_data[$client_ip]['count'];
            $login_error = "Wrong password. " . max(0, $remaining_attempts) . " attempts remaining.";
        }
    }
}

// Enhanced approve with validation
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['approve_id']) && !empty($_SESSION['is_admin'])) {
    $approve_id = trim($_POST['approve_id']);
    $admin_note = trim($_POST['admin_note'] ?? '');
    
    if (empty($approve_id)) {
        $error_message = "Invalid request ID";
    } else {
        $file = __DIR__ . '/requests.json';
        $data = json_decode(@file_get_contents($file) ?: '[]', true);
        
        $found = false;
        foreach ($data as &$req) {
            if ($req['id'] === $approve_id && $req['status'] === 'pending') {
                $req['status'] = 'approved';
                $req['admin_note'] = $admin_note;
                $req['approved_by'] = 'admin';
                $req['approved_at'] = gmdate('Y-m-d H:i:s');
                $req['approved_at_gmt6'] = gmdate('Y-m-d H:i:s') . ' +06';
                $found = true;
                
                // Log the approval
                $log_entry = [
                    'action' => 'approved',
                    'request_id' => $approve_id,
                    'admin_note' => $admin_note,
                    'timestamp' => gmdate('Y-m-d H:i:s'),
                    'ip' => $_SERVER['REMOTE_ADDR']
                ];
                $log_file = __DIR__ . '/admin_actions.log';
                file_put_contents($log_file, json_encode($log_entry) . "\n", FILE_APPEND);
                break;
            }
        }
        
        if ($found) {
            file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES));
            $success_message = "Request approved successfully";
        } else {
            $error_message = "Request not found or already processed";
        }
    }
    
    header("Location: " . $_SERVER['REQUEST_URI'] . ($error_message ? "?error=" . urlencode($error_message) : "?success=" . urlencode($success_message)));
    exit;
}

// mark processed (existing functionality - now works with approved requests)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['process_id']) && !empty($_SESSION['is_admin'])) {
    $process_id = $_POST['process_id'];
    $admin_note = $_POST['admin_note'] ?? '';
    
    $file = __DIR__ . '/requests.json';
    $data = json_decode(@file_get_contents($file) ?: '[]', true);
    
    // Find the request by ID
    foreach ($data as &$req) {
        if ($req['id'] === $process_id) {
            $req['status'] = 'processed';
            $req['admin_note'] = $admin_note;
            $req['processed_by'] = 'admin';
            $req['processed_at'] = gmdate('Y-m-d H:i:s');
            $req['processed_at_gmt6'] = gmdate('Y-m-d H:i:s') . ' +06';
            break;
        }
    }
    
    file_put_contents($file, json_encode($data, JSON_PRETTY_PRINT|JSON_UNESCAPED_SLASHES));
    header("Location: ".$_SERVER['REQUEST_URI']);
    exit;
}

if (empty($_SESSION['is_admin'])) {
    // show password form
    ?>
    <!doctype html><html><head><meta name="viewport" content="width=device-width,initial-scale=1"><title>Admin Login</title>
    <style>body{font-family:Inter,Arial;padding:20px;background:#f3f4f6} .card{max-width:420px;margin:30px auto;background:#fff;padding:18px;border-radius:12px;box-shadow:0 6px 18px rgba(2,6,23,.08)}</style>
    </head><body><div class="card">
    <h2>Admin Login</h2>
    <?php if(!empty($login_error)) echo "<div style='color:#b91c1c;margin-bottom:8px'>$login_error</div>"; ?>
    <form method="post">
      <label>Password</label>
      <input name="admin_pass" type="password" style="width:100%;padding:10px;margin-top:6px;border-radius:8px;border:1px solid #e6edf3">
      <div style="margin-top:10px"><button style="padding:10px 12px;background:#111827;color:#fff;border:0;border-radius:8px">Login</button></div>
    </form>
    </div></body></html>
    <?php
    exit;
}

// load requests and sort by created_at (newest first)
$file = __DIR__ . '/requests.json';
$data = json_decode(@file_get_contents($file) ?: '[]', true);
if (!is_array($data)) $data = [];

// Sort by created_at in descending order (newest first)
usort($data, function($a, $b) {
    return strtotime($b['created_at']) - strtotime($a['created_at']);
});

// Function to get status badge color
function getStatusColor($status) {
    switch($status) {
        case 'pending': return 'background:#fef3c7;color:#92400e';
        case 'approved': return 'background:#d1fae5;color:#065f46';
        case 'processed': return 'background:#ecfccb;color:#365314';
        default: return 'background:#f3f4f6;color:#374151';
    }
}
?>
<!doctype html>
<html>
<head>
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>Admin - Code Change Requests</title>
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;500;700;800&display=swap" rel="stylesheet">
<style>
body{font-family:Inter,system-ui;padding:12px;background:#f8fafc;color:#0f172a}
.container{max-width:920px;margin:0 auto}
.header{display:flex;justify-content:space-between;align-items:center;margin-bottom:12px}
.card{background:#fff;border-radius:12px;padding:14px;margin-bottom:12px;box-shadow:0 6px 18px rgba(15,23,42,0.06);display:flex;gap:12px;align-items:center}
.left{flex:1}
.tag{padding:6px 12px;border-radius:999px;font-weight:700;font-size:12px}
.small{font-size:13px;color:#6b7280}
.btn{padding:8px 12px;border-radius:8px;border:0;font-weight:700;cursor:pointer;font-size:12px}
.copy{background:#10b981;color:#fff}
.screen{background:#111827;color:#fff}
.approve{background:#f59e0b;color:#fff}
.process{background:#2563eb;color:#fff}
.reject{background:#dc2626;color:#fff}
.icon{width:54px;height:54px;border-radius:12px;background:linear-gradient(135deg,#ecfeff,#dbeafe);display:flex;align-items:center;justify-content:center;font-weight:800;color:#0f172a}
.card:hover{transform:translateY(-4px);transition:all .25s ease}
.admin-note{margin-top:8px;font-size:13px;padding:8px;background:#f8fafc;border-radius:6px}
.form-inline{display:inline-flex;gap:8px;align-items:center}
.note-input{padding:6px 8px;border-radius:6px;border:1px solid #e6edf3;font-size:12px;width:150px}
.timestamps{margin-top:6px;font-size:11px;color:#9ca3af}
</style>
</head>
<body>
<div class="container">
  <div class="header">
    <h2>Code Change Requests</h2>
    <div class="small">
      Logged in as admin | Total: <?= count($data) ?> | 
      Session expires: <?= date('H:i:s', $_SESSION['last_activity'] + $SESSION_TIMEOUT) ?>
      <a href="?logout=1" style="margin-left:10px;color:#dc2626;text-decoration:none">[Logout]</a>
    </div>
  </div>
  
  <div class="nav" style="display:flex;gap:10px;margin-bottom:20px;flex-wrap:wrap">
    <a href="admin.php" style="text-decoration:none;padding:8px 16px;background:<?= empty($_GET['filter']) ? '#2563eb' : '#fff' ?>;color:<?= empty($_GET['filter']) ? '#fff' : '#0f172a' ?>;border-radius:8px;font-weight:500">All (<?= count($data) ?>)</a>
    <a href="?filter=pending" style="text-decoration:none;padding:8px 16px;background:<?= $_GET['filter'] === 'pending' ? '#f59e0b' : '#fff' ?>;color:<?= $_GET['filter'] === 'pending' ? '#fff' : '#0f172a' ?>;border-radius:8px;font-weight:500">Pending (<?= count(array_filter($data, function($r) { return $r['status'] === 'pending'; })) ?>)</a>
    <a href="?filter=approved" style="text-decoration:none;padding:8px 16px;background:<?= $_GET['filter'] === 'approved' ? '#10b981' : '#fff' ?>;color:<?= $_GET['filter'] === 'approved' ? '#fff' : '#0f172a' ?>;border-radius:8px;font-weight:500">Approved (<?= count(array_filter($data, function($r) { return $r['status'] === 'approved'; })) ?>)</a>
    <a href="?filter=processed" style="text-decoration:none;padding:8px 16px;background:<?= $_GET['filter'] === 'processed' ? '#22c55e' : '#fff' ?>;color:<?= $_GET['filter'] === 'processed' ? '#fff' : '#0f172a' ?>;border-radius:8px;font-weight:500">Processed (<?= count(array_filter($data, function($r) { return $r['status'] === 'processed'; })) ?>)</a>
  </div>

  <?php 
  // Handle logout
  if (isset($_GET['logout'])) {
      session_destroy();
      header("Location: admin.php");
      exit;
  }
  
  // Show messages
  if (isset($_GET['error'])): ?>
    <div style="background:#fef2f2;border:1px solid #fecaca;color:#b91c1c;padding:10px;border-radius:8px;margin-bottom:15px">
      <?= htmlspecialchars($_GET['error']) ?>
    </div>
  <?php endif; ?>
  
  <?php if (isset($_GET['success'])): ?>
    <div style="background:#f0fdf4;border:1px solid #bbf7d0;color:#166534;padding:10px;border-radius:8px;margin-bottom:15px">
      <?= htmlspecialchars($_GET['success']) ?>
    </div>
  <?php endif; ?>

  <?php 
  // Filter requests if needed
  $filtered_data = $data;
  if (isset($_GET['filter']) && $_GET['filter'] === 'pending') {
      $filtered_data = array_filter($data, function($req) { return $req['status'] === 'pending'; });
  }
  elseif (isset($_GET['filter']) && $_GET['filter'] === 'approved') {
      $filtered_data = array_filter($data, function($req) { return $req['status'] === 'approved'; });
  }
  elseif (isset($_GET['filter']) && $_GET['filter'] === 'processed') {
      $filtered_data = array_filter($data, function($req) { return $req['status'] === 'processed'; });
  }
  ?>

  <?php if (count($filtered_data) === 0): ?>
    <div class="card">
      <div class="left">
        <div style="font-weight:700">No requests</div>
        <div class="small">No requests found matching the current filter.</div>
      </div>
    </div>
  <?php else: ?>
    <?php foreach ($filtered_data as $req): ?>
      <div class="card" id="card-<?= htmlspecialchars($req['id']) ?>">
        <div class="icon"><?= htmlspecialchars(substr($req['current_code'], 0, 1)) ?></div>
        <div class="left">
          <div style="display:flex;justify-content:space-between;align-items:center">
            <div>
              <div style="font-weight:800">
                <?= htmlspecialchars($req['current_code']) ?> → 
                <span style="color:#0f172a"><?= htmlspecialchars($req['new_code']) ?></span>
              </div>
              <div class="small">
                Mobile: <?= htmlspecialchars($req['mobile']) ?> • 
                Room: <?= htmlspecialchars($req['room']) ?> • 
                ID: <?= htmlspecialchars(substr($req['id'], -8)) ?>
              </div>
              <div class="timestamps">
                Created: <?= htmlspecialchars($req['created_at_gmt6']) ?>
                <?php if (isset($req['approved_at'])): ?>
                  <br>Approved: <?= htmlspecialchars($req['approved_at_gmt6']) ?>
                <?php endif; ?>
                <?php if (isset($req['processed_at'])): ?>
                  <br>Processed: <?= htmlspecialchars($req['processed_at_gmt6']) ?>
                <?php endif; ?>
              </div>
            </div>
            <div>
              <div style="text-align:right">
                <div class="tag" style="<?= getStatusColor($req['status']) ?>">
                  <?= ucfirst(htmlspecialchars($req['status'])) ?>
                </div>
              </div>
            </div>
          </div>

          <?php if (!empty($req['admin_note'])): ?>
            <div class="admin-note">
              <strong>Admin Note:</strong> <?= htmlspecialchars($req['admin_note']) ?>
            </div>
          <?php endif; ?>

          <div style="margin-top:10px;display:flex;gap:8px;flex-wrap:wrap">
            <button class="btn copy" data-new="<?= htmlspecialchars($req['new_code']) ?>">Copy New Code</button>
            <button class="btn screen" data-card="#card-<?= htmlspecialchars($req['id']) ?>">Screenshot</button>

            <?php if ($req['status'] === 'pending'): ?>
              <!-- Approve Button -->
              <form method="post" class="form-inline">
                <input type="hidden" name="approve_id" value="<?= htmlspecialchars($req['id']) ?>">
                <input type="text" name="admin_note" placeholder="Approval note (optional)" class="note-input">
                <button class="btn approve" type="submit">Approve</button>
              </form>
              
            <?php elseif ($req['status'] === 'approved'): ?>
              <!-- Mark as Processed Button -->
              <form method="post" class="form-inline">
                <input type="hidden" name="process_id" value="<?= htmlspecialchars($req['id']) ?>">
                <input type="text" name="admin_note" placeholder="Process note (optional)" class="note-input">
                <button class="btn process" type="submit">Mark Processed</button>
              </form>
              
            <?php else: ?>
              <!-- Already Processed -->
              <div style="padding:8px 12px;border-radius:8px;background:#ecfccb;color:#365314;font-weight:700;font-size:12px">
                ✓ Completed
              </div>
            <?php endif; ?>
          </div>
        </div>
      </div>
    <?php endforeach; ?>
  <?php endif; ?>

  <div style="margin-top:20px;padding:12px;background:#fff;border-radius:8px;border-left:4px solid #2563eb" class="small">
    <strong>Workflow:</strong> 
    <span style="color:#92400e">Pending</span> → 
    <span style="color:#065f46">Approved</span> → 
    <span style="color:#365314">Processed</span>
    <br><br>
    <strong>Tips:</strong><br>
    • Requests are sorted with newest first<br>
    • Use "Copy New Code" to quickly get the new code value<br>
    • Screenshot feature helps with record keeping<br>
    • Add notes during approval/processing for audit trail
  </div>
</div>

<script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
<script>
document.addEventListener('click', function(e){
  if (e.target && e.target.classList.contains('copy')) {
    const code = e.target.dataset.new;
    navigator.clipboard.writeText(code).then(()=> {
      e.target.innerText = 'Copied ✓';
      setTimeout(()=> e.target.innerText = 'Copy New Code', 1400);
    });
  }
  if (e.target && e.target.classList.contains('screen')) {
    const selector = e.target.dataset.card;
    const el = document.querySelector(selector);
    if (!el) return;
    html2canvas(el).then(canvas => {
      const a = document.createElement('a');
      a.href = canvas.toDataURL('image/png');
      a.download = 'request-'+Date.now()+'.png';
      a.click();
      e.target.innerText = 'Saved ✓';
      setTimeout(()=> e.target.innerText = 'Screenshot', 1400);
    });
  }
});

// Auto-refresh every 30 seconds, but only if no forms are being filled
let refreshTimer;
let formActivity = false;

function resetRefreshTimer() {
  clearTimeout(refreshTimer);
  refreshTimer = setTimeout(() => {
    if (!formActivity) location.reload();
  }, 30000);
}

// Monitor form activity
document.addEventListener('focusin', function(e) {
  if (e.target.tagName === 'INPUT' || e.target.tagName === 'TEXTAREA') {
    formActivity = true;
    clearTimeout(refreshTimer);
  }
});

document.addEventListener('focusout', function(e) {
  if (e.target.tagName === 'INPUT' || e.target.tagName === 'TEXTAREA') {
    formActivity = false;
    resetRefreshTimer();
  }
});

// Start the timer
resetRefreshTimer();
</script>
</body>
</html>