#!/usr/bin/env python3
import json
import logging
import paramiko
from datetime import datetime
import os
import uuid

# ==============================
# Logging setup
# ==============================
# Create logger
logger = logging.getLogger()
logger.setLevel(logging.DEBUG)

# Create file handler which logs even debug messages
fh = logging.FileHandler('mikrotik_processor.log')
fh.setLevel(logging.DEBUG)
fh_formatter = logging.Formatter('%(asctime)s - %(levelname)s - %(message)s')
fh.setFormatter(fh_formatter)
logger.addHandler(fh)

# Create console handler with a higher log level
ch = logging.StreamHandler()
ch.setLevel(logging.INFO)
ch_formatter = logging.Formatter('%(message)s')
ch.setFormatter(ch_formatter)
logger.addHandler(ch)

# ==============================
# Config
# ==============================
# Use local paths for testing
CONFIG_FILE = os.path.join(os.path.dirname(os.path.abspath(__file__)), "config.json")
REQUESTS_FILE = os.path.join(os.path.dirname(os.path.abspath(__file__)), "requests.json")
PROCESSED_FILE = os.path.join(os.path.dirname(os.path.abspath(__file__)), "processed_log.json")

# Flag to determine whether to preserve processed requests in the requests.json file
# Set to True to keep processed requests in the file, False to remove them
PRESERVE_PROCESSED_REQUESTS = True

# ==============================
# Helper functions
# ==============================
def load_json(file_path):
    if not os.path.exists(file_path):
        logging.warning(f"File not found: {file_path}, returning empty list")
        return []
    try:
        with open(file_path, "r") as f:
            data = json.load(f)
            logging.debug(f"Loaded {len(data)} entries from {file_path}")
            return data
    except Exception as e:
        logging.error(f"Failed to load {file_path}: {e}")
        return []

def save_json(file_path, data):
    try:
        logging.debug(f"Attempting to save to {file_path}")
        logging.debug(f"File path exists: {os.path.exists(os.path.dirname(file_path))}")
        logging.debug(f"Data to save: {json.dumps(data)}")
        
        with open(file_path, "w") as f:
            json.dump(data, f, indent=2)
        
        # Verify the file was updated
        if os.path.exists(file_path):
            with open(file_path, "r") as f:
                saved_data = json.load(f)
            logging.debug(f"Verification - Read back {len(saved_data)} entries from {file_path}")
        else:
            logging.error(f"Verification failed - File {file_path} does not exist after save attempt")
            
        logging.debug(f"Saved {len(data)} entries to {file_path}")
    except Exception as e:
        logging.error(f"Failed to save {file_path}: {e}")
        import traceback
        logging.error(traceback.format_exc())

def run_mikrotik_commands(host, port, user, password, commands):
    logging.info(f"Connecting to MikroTik {host}:{port} as {user}")
    
    # For testing purposes, simulate success without actual SSH connection
    try:
        # Log commands that would be executed
        for cmd in commands:
            logging.info(f"Would run command: {cmd}")
        
        logging.info("TEST MODE: Simulating successful command execution")
        return True
        
        # Real implementation (commented out for testing)
        '''
        ssh = paramiko.SSHClient()
        ssh.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        ssh.connect(host, port=port, username=user, password=password, timeout=10)
        for cmd in commands:
            logging.info(f"Running command: {cmd}")
            stdin, stdout, stderr = ssh.exec_command(cmd)
            out = stdout.read().decode().strip()
            err = stderr.read().decode().strip()
            logging.debug(f"Command output: {out if out else '[no output]'}")
            if err:
                logging.warning(f"Command error: {err}")
        ssh.close()
        logging.info("SSH connection closed")
        return True
        '''
    except Exception as e:
        logging.error(f"SSH command failed: {e}")
        return False

# ==============================
# Main script
# ==============================
if __name__ == "__main__":
    logging.info("=== MikroTik Request Processor Started ===")

    # Load config
    if not os.path.exists(CONFIG_FILE):
        logging.critical(f"Config file {CONFIG_FILE} not found!")
        exit(1)

    with open(CONFIG_FILE) as f:
        config = json.load(f)
    mikrotik_cfg = config.get("mikrotik", {})
    if not mikrotik_cfg:
        logging.critical("MikroTik config missing in config.json!")
        exit(1)

    logging.debug(f"Loading requests from {REQUESTS_FILE}")
    requests = load_json(REQUESTS_FILE)
    logging.debug(f"Loaded {len(requests)} requests")
    
    logging.debug(f"Loading processed records from {PROCESSED_FILE}")
    processed = load_json(PROCESSED_FILE)
    logging.debug(f"Loaded {len(processed)} processed records")

    if not requests:
        logging.info("No pending requests to process.")
        exit(0)
        
    logging.debug(f"Found {len(requests)} requests to process")

    # Create a copy of the original requests for preservation if needed
    original_requests = requests.copy() if PRESERVE_PROCESSED_REQUESTS else []
    
    # Track which requests were successfully processed
    processed_request_ids = []
    
    for req in requests[:]:  # copy list to modify while iterating
        # Ensure ID exists
        if "id" not in req or not req["id"]:
            req["id"] = f"req_{uuid.uuid4()}"

        # Handle old/current code field names
        old_code = req.get('old_code') or req.get('current_code')
        new_code = req.get('new_code')

        if not old_code or not new_code:
            logging.error(f"Skipping request {req['id']} due to missing codes: {req}")
            continue

        logging.info(
            f"Processing request ID={req['id']} OldCode={old_code} NewCode={new_code}"
        )

        commands = [
            f'/user-manager session remove [find where user="{old_code}"]',
            f'/user-manager user set [find where name="{old_code}"] name={new_code}'
        ]
        success = run_mikrotik_commands(
            mikrotik_cfg["host"],
            mikrotik_cfg.get("port", 22),
            mikrotik_cfg["username"],
            mikrotik_cfg["password"],
            commands
        )

        if success:
            logging.debug(f"Request ID={req['id']} successful, adding to processed list")
            processed.append(req)
            
            # Track this request ID as processed
            processed_request_ids.append(req['id'])
            
            # Only remove from requests list if we're not preserving processed requests
            if not PRESERVE_PROCESSED_REQUESTS:
                logging.debug(f"Request ID={req['id']} removing from requests list")
                logging.debug(f"Requests before removal: {len(requests)}")
                requests.remove(req)
                logging.debug(f"Requests after removal: {len(requests)}")
            
            logging.info(f"Request ID={req['id']} processed successfully")
        else:
            logging.error(f"Request ID={req['id']} failed, leaving in queue")

    # Create a backup of processed requests before clearing the file
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    backup_file = os.path.join(os.path.dirname(os.path.abspath(__file__)), f"requests_backup_{timestamp}.json")
    
    # Save backup of original requests
    original_requests = load_json(REQUESTS_FILE)
    save_json(backup_file, original_requests)
    logging.info(f"Created backup of requests at {backup_file}")
    
    # Determine what to save back to the requests file
    if PRESERVE_PROCESSED_REQUESTS:
        # If preserving processed requests, we need to update the status of processed requests
        logging.info("Preserving processed requests in the requests.json file with updated status")
        
        # Update the status of processed requests in the original_requests list
        for req in original_requests:
            if req.get('id') in processed_request_ids:
                req['status'] = 'processed'
                logging.debug(f"Updated status of request ID={req['id']} to 'processed'")
        
        # Save the updated original requests back to the file
        logging.debug(f"Saving {len(original_requests)} requests (including processed) back to {REQUESTS_FILE}")
        save_json(REQUESTS_FILE, original_requests)
    else:
        # If not preserving, save only the unprocessed requests
        logging.debug(f"Saving {len(requests)} unprocessed requests back to {REQUESTS_FILE}")
        save_json(REQUESTS_FILE, requests)
    
    logging.debug(f"Saving {len(processed)} processed records to {PROCESSED_FILE}")
    save_json(PROCESSED_FILE, processed)

    # Verify files after save
    logging.debug("Verifying files after save:")
    logging.debug(f"Requests file exists: {os.path.exists(REQUESTS_FILE)}")
    logging.debug(f"Processed file exists: {os.path.exists(PROCESSED_FILE)}")
    
    if os.path.exists(REQUESTS_FILE):
        with open(REQUESTS_FILE, 'r') as f:
            final_requests = json.load(f)
        logging.debug(f"Final requests count: {len(final_requests)}")
    
    logging.info("=== MikroTik Request Processor Finished ===")
